﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Windows.Forms;
using File_Utility;

/// <summary>
/// The classes inside this script are responsible for handling different functionality for the temporary data base. The state machine is implemented in the non-static classes to provide overwrite functionalitys for lines stored within the database.
/// Static class provides different functionality for the data base. Such as adding lines and resetting the data base. 
/// The core data structures that are used to serialize and deseralize XML files are also defined within this script.
/// 
/// Author: Sam Scott
/// </summary>
namespace GenerateLine
{
    /// <summary>
    /// These classes use a state based architecture to allow either a new line entry to be added, or an existed entry to be overridden and replaced
    /// </summary>

    namespace GenerateLine.State
    {
        public enum LineState { Default, Overwritting };
        public abstract class LineWritingState
        {
            protected LineWritingState state;

            public static LineState lineState;
            public abstract void GenerateLine(LineData lineData);
            public abstract void ChangeState(bool isOverwritting);
            public abstract void ResetState();
        }

        public class GenerateNewLine : LineWritingState
        {

            public GenerateNewLine(LineWritingState _state, LineState _lineState)
            {
                this.state = _state;
                // lineState = _lineState;

            }
            public override void ChangeState(bool isOverwritting)
            {
                if (isOverwritting)
                {
                    lineState = LineState.Overwritting;

                }
            }

            public override void GenerateLine(LineData lineData)
            {
                if (lineState == LineState.Default)
                {
                    Console.WriteLine("DefaultState: ");
                    LineHandling.AddLineDataToLists(lineData); //Add populated object to the temporary object list
                }

                else
                {
                    lineState = LineState.Overwritting;
                    this.state = new OverwriteExistingLine(this, lineState);
                    this.state.GenerateLine(lineData);
                }
            }
            public override void ResetState()
            {
                //Do nothing

            }
        }

        public class OverwriteExistingLine : LineWritingState
        {

            public OverwriteExistingLine(LineWritingState _state, LineState _lineState)
            {
                Console.WriteLine("New Overwrite State");
            }
            public override void ChangeState(bool isOverwritting)
            {
                Console.WriteLine("Overwritestate");
            }

            public override void GenerateLine(LineData lineData)
            {
                if (lineState == LineState.Overwritting)
                {
                    Console.WriteLine("State: " + lineState);

                    LineHandling.ReplaceLineFromList(lineData);

                    ResetState();
                }
                else
                {
                    this.state = new GenerateNewLine(this, lineState);
                    this.state.GenerateLine(lineData);
                }

            }

            public override void ResetState()
            {
                //Set state back to normal
                lineState = LineState.Default;
            }
        }

        /// <summary>
        /// This class is used to handle the Line data collection that acts as a seperate collection from UI collections, so the user can edit elements 
        /// </summary>
        public static class LineHandling
        {
            static List<LineData> temporaryList;

            static int currentIndex;

            private const string keyListKVPName = "KeyListBox";
            private const string lineListKVPName = "LineListBox";
            private const string conditionsAmountListKVPName = "ConditionsAmountListBox";
            private const string gameDataNameListKVPName = "GameDataNameListBox";
            private const string gameDataKeyListKVPName = "GameDataKeyListBox";
            private const string designerKeyListKVPName = "DeisngerKeyListBox";


            static Dictionary<string, ListBox> lineListsDict;

            public static void Initialise(ListBox keyListBox, ListBox lineListBox, ListBox conditionsAmountListBox, ListBox gameDataNameListBox, ListBox gameDataKeyListBox, ListBox designerKeyListBox)
            {
                temporaryList = new List<LineData>(); //List to hold all created lines for the session

                //Set up a link between form lists and line handling functioality 
                lineListsDict = new Dictionary<string, ListBox>
                {
                    { keyListKVPName, keyListBox },
                    { lineListKVPName, lineListBox },
                    { conditionsAmountListKVPName, conditionsAmountListBox },
                    { gameDataNameListKVPName, gameDataNameListBox},
                    { gameDataKeyListKVPName, gameDataKeyListBox },
                    { designerKeyListKVPName, designerKeyListBox}
                };
            }


            /// <summary>
            /// Adds newly created line to lineData database
            /// </summary>
            /// <param name="lineToStore"> object containing all var values for the line</param>
            public static void AddLineDataToLists(LineData lineToStore)
            {
                ListBox keyList;
                lineListsDict.TryGetValue(keyListKVPName, out keyList);

                ListBox lineList;
                lineListsDict.TryGetValue(lineListKVPName, out lineList);

                ListBox conditionsList;
                lineListsDict.TryGetValue(conditionsAmountListKVPName, out conditionsList);


                keyList.Items.Add(lineToStore.key);

                lineList.Items.Add(lineToStore.lineText);


                if (lineToStore.lineConditionsDictionary != null)
                {
                    conditionsList.Items.Add(lineToStore.lineConditionsDictionary.Count);
                }

                else
                {
                    conditionsList.Items.Add(0);
                }

                temporaryList.Add(lineToStore);
                Console.WriteLine("Temp List Count: " + temporaryList.Count);
            }

            /// <summary>
            // Replaces line from list
            /// </summary>
            /// <param name="lineData">Line to replace with</param>
            public static void ReplaceLineFromList(LineData lineData)
            {

                ListBox keyList;
                lineListsDict.TryGetValue(keyListKVPName, out keyList);

                ListBox lineList;
                lineListsDict.TryGetValue(lineListKVPName, out lineList);

                ListBox conditionsList;
                lineListsDict.TryGetValue(conditionsAmountListKVPName, out conditionsList);

                keyList.Items[currentIndex] = lineData.key;

                lineList.Items[currentIndex] = lineData.lineText;


                conditionsList.Items[currentIndex] = lineData.lineConditionsDictionary.Count;

                temporaryList[currentIndex] = lineData;

            }

            public static void CreateXMLFile(string id, string name, string fileName, string directory, string root)
            {
                FileHandler.CreateXmlFile(id, name, temporaryList, fileName, directory, root);
            }

            public static void Test()
            {
                ListBox keyList;
                lineListsDict.TryGetValue("KeyListBox", out keyList);

                keyList.Items[currentIndex] = "Hello";
            }

            public static void DeleteLine(int index)
            {
                temporaryList.RemoveAt(index);
            }

            public static LineData GetLineData(int index)
            {
                currentIndex = index;
                return temporaryList[index];
            }

            private static void ResetDataBase()
            {
                //Reset all lists, before we load from file
                temporaryList.Clear();

                lineListsDict[keyListKVPName].Items.Clear();
                lineListsDict[lineListKVPName].Items.Clear();
                lineListsDict[conditionsAmountListKVPName].Items.Clear();
                lineListsDict[gameDataNameListKVPName].Items.Clear();
                lineListsDict[gameDataKeyListKVPName].Items.Clear();
                lineListsDict[designerKeyListKVPName].Items.Clear();


            }

            public static void SetUpDataBaseFromFile(List<LineData> lineData)
            {
                ResetDataBase();

                foreach (var line in lineData)
                {
                    AddLineDataToLists(line);
                }
            }
        }
    }
}

//This object is used for deserialization only!
public class NPCData
{
    public uint npcID;
    public string npcName;
}

// Data we want to contain into an object for easy viewing ect, line per line basis 
public class LineData
{
    public uint lineID;
    public string key;
    public string lineText;
    public Dictionary<uint, LineConditionData> lineConditionsDictionary;
}


//Conditions require their own entity, which can be stored in a dictionary for each line data (entity line)
public class LineConditionData
{
    public string gameDataName;
    public string lineCondition;
    public uint gameDataKey;
    public uint designerKey;
}
