﻿using System;
using System.Collections.Generic;
using System.Windows.Forms;
using File_Utility;
using GenerateLine.GenerateLine.State;
using Validation.Input;
using System.IO;


namespace Dissertation_EntityDialogueSystem_XMLForm
{
    /// <summary>
    /// UI Handling for the XML generation system, provides events for different form UI elements e.g. button press, text box entered
    /// Author: Sam Scott
    /// </summary>
    public partial class Form1 : Form
    {
        private LineWritingState lineWritingState;


        public Form1() //Constructor
        {
            InitializeComponent();
            lineWritingState = new GenerateNewLine(lineWritingState, LineState.Default); //Set up initial file writing state to setting new state 
            LineHandling.Initialise(keysList, linesList, conditionsAmountList, GameDataNameList, GameDataKeyListBox, designerKeysListBox); // Set up temp dictionary that contains references to all necessary list boxes that users interacts with in this script
        }

        //Create file
        private void button1_MouseClick(object sender, MouseEventArgs e)
        {
            Console.WriteLine("ID: " + idText.Text);
            Console.WriteLine("Name: " + nameText.Text);
            Console.WriteLine("Key: " + keyText.Text);
            Console.WriteLine("Line: " + lineText.Text);
            LineHandling.CreateXMLFile(idText.Text, nameText.Text, fileNameText.Text, dirText.Text, xmlRootNameText.Text);
        }


        /// <summary>
        /// Add all condition fields to create a new line condition entry
        /// </summary>
        private void AddCondition()
        {
            lineConditionsList.Items.Add(LineConditionText.Text); //Add condition to the current lines condition list box
            GameDataKeyListBox.Items.Add(gameDataKeyTextBox.Text);
            GameDataNameList.Items.Add(gameDataTextBox.Text);
            designerKeysListBox.Items.Add(designerKeyTextBox.Text);
            LineConditionText.Clear();
            gameDataKeyTextBox.Clear();
            gameDataTextBox.Clear();
            designerKeyTextBox.Clear();
        }


        /// <summary>
        /// Triggered when the line condition add button is clicked, will ensure condition is a correct type and that all fields are valid
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        //Add line condition button clicked
        private void AddLineConditionButton_Click(object sender, EventArgs e)
        {
            if (LineConditionText.Text != "" && gameDataTextBox.Text != "" && gameDataKeyTextBox.Text != "" && designerKeyTextBox.Text != "" && InputValidation.CheckInputtedConditionIsValid(LineConditionText))
            {
                AddCondition();
            }
            else if (!InputValidation.CheckInputtedConditionIsValid(LineConditionText) && LineConditionText.Text != "" && gameDataTextBox.Text != "" && gameDataKeyTextBox.Text != "" && designerKeyTextBox.Text != "")
            {
                MessageBox.Show("Not a supported data type for condition value. Please make it of: bool (true, false), int (e.g. 1) or float e.g. (1.6)");
            }
            else
            {
                MessageBox.Show("Please fill in all fields relating to this line condition!");
            }
        }


        /// <summary>
        /// Adds the line to the line data base
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void AddLineButton_Click(object sender, EventArgs e) //Add line
        {

            if (keyText.Text != "" && lineText != null)
            {
                //Create an element of the temporary line list
                LineData userData = new LineData
                {
                    lineID = uint.Parse(lineIDText.Text),
                    key = keyText.Text,
                    lineText = lineText.Text,
                    lineConditionsDictionary = new Dictionary<uint, LineConditionData>()

                };


                //Iterate over conditions list box for line and add it to the object list 
                for (int i = 0; i < lineConditionsList.Items.Count; i++)
                {
                    //Create new instance of lineconditions object
                    LineConditionData conditionData = new LineConditionData();


                    //Populate object based on list boxes then add to Line object 
                    uint dataKey = uint.Parse(GameDataKeyListBox.Items[i].ToString());
                    conditionData.gameDataKey = dataKey;
                    conditionData.gameDataName = GameDataNameList.Items[i].ToString();
                    conditionData.lineCondition = lineConditionsList.Items[i].ToString();
                    conditionData.designerKey = uint.Parse(designerKeysListBox.Items[i].ToString());
                    userData.lineConditionsDictionary.Add(conditionData.gameDataKey, conditionData); //RandomKey should probs implement sound designer key as current game key

                }

                lineWritingState.GenerateLine(userData);
                lineWritingState.ChangeState(false);
                ClearAllLineInputFields();
            }
            else
            {
                MessageBox.Show("Please ensure each line has filled key and line field!");
            }
        }

        /// <summary>
        /// Deletes condition from a line being created
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void DeleteConditionButton_Click(object sender, EventArgs e)
        {
            int indexToRemove = lineConditionsList.SelectedIndex;
            lineConditionsList.Items.RemoveAt(indexToRemove);
            GameDataNameList.Items.RemoveAt(indexToRemove); //NOT WORKING>??
            GameDataKeyListBox.Items.RemoveAt(indexToRemove); //NOT WORKING???
            designerKeysListBox.Items.RemoveAt(indexToRemove);

        }


        /// <summary>
        /// Deletes line from line database
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void DeleteLineButton_Click(object sender, EventArgs e)
        {
            if (linesList.SelectedItem != null)
            {
                ClearLine(linesList.SelectedIndex);
            }
        }


        /// <summary>
        /// Clears all entries releating to a single line
        /// </summary>
        /// <param name="indexSelected"></param>
        private void ClearLine(int indexSelected)
        {

            linesList.Items.RemoveAt(indexSelected);
            keysList.Items.RemoveAt(indexSelected);
            conditionsAmountList.Items.RemoveAt(indexSelected);
            LineHandling.DeleteLine(indexSelected);

        }


        /// <summary>
        /// Re-populates line fields with data relating to selected line
        /// </summary>
        private void ShowSelectedLineData()
        {
            ClearAllLineInputFields();

            LineData line = LineHandling.GetLineData(linesList.SelectedIndex); //Get line at selected index from temporary list of all lines

            lineIDText.Text = line.lineID.ToString();
            keyText.Text = line.key;
            lineText.Text = line.lineText;

            if (line.lineConditionsDictionary != null)
            {
                foreach (LineConditionData condition in line.lineConditionsDictionary.Values)
                {
                    GameDataKeyListBox.Items.Add(condition.gameDataKey);
                    GameDataNameList.Items.Add(condition.gameDataName);
                    lineConditionsList.Items.Add(condition.lineCondition);
                    designerKeysListBox.Items.Add(condition.designerKey);
                }
            }


        }

        /// <summary>
        /// Replaces edited line from selected list element
        /// </summary>
        /// <param name="lineData"></param>
        /// <param name="index"></param>
        public void ReplaceLineListBoxElement(LineData lineData, int index)
        {
            linesList.Items[index] = lineData.lineText;
            keysList.Items[index] = lineData.key;

            int i = 0;
            foreach (LineConditionData condition in lineData.lineConditionsDictionary.Values)
            {
                lineConditionsList.Items[i] = condition.lineCondition; //May be wrong was a bit different before change
                GameDataKeyListBox.Items[i] = condition.gameDataKey;
                GameDataNameList.Items[i] = condition.gameDataName;
                designerKeysListBox.Items[i] = condition.designerKey;
                i++;
            }
        }


        private void ClearAllLineInputFields()
        {
            //Clear text from all line related input fields
            keyText.Clear();
            lineText.Clear();
            lineIDText.Clear();
            LineConditionText.Clear();
            lineConditionsList.Items.Clear();
            GameDataKeyListBox.Items.Clear();
            GameDataNameList.Items.Clear();
            gameDataKeyTextBox.Clear();
            gameDataTextBox.Clear();
            designerKeysListBox.Items.Clear();
            designerKeyTextBox.Clear();
        }

        private void EditLineButton_Click(object sender, EventArgs e)
        {
            if (linesList.SelectedItem != null)
            {
                lineWritingState.ChangeState(true);
                ShowSelectedLineData();
            }
        }

        /// Testing purposes /////
        private void TestButton_Click(object sender, EventArgs e)
        {
            LineHandling.Test();
        }

        private void IdText_TextChanged(object sender, EventArgs e)
        {
            InputValidation.CheckInputedKeyIsValid(idText);
        }
        private void GameDataKeyTextBox_TextChanged(object sender, EventArgs e)
        {
            InputValidation.CheckInputedKeyIsValid(gameDataKeyTextBox);
        }

        private void LineIDText_TextChanged(object sender, EventArgs e)
        {
            InputValidation.CheckInputedKeyIsValid(lineIDText);
        }

        private void DesignerKeyTextBox_TextChanged(object sender, EventArgs e)
        {
            InputValidation.CheckInputedKeyIsValid(designerKeyTextBox);
        }


        /// <summary>
        /// Used to search for a folder and get the path to save the inputed data to file
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BrowseForFolderButton_Click(object sender, EventArgs e)
        {
            FolderBrowserDialog folderDlg = new FolderBrowserDialog();
            folderDlg.ShowNewFolderButton = true;
            // Show the FolderBrowserDialog.  
            DialogResult result = folderDlg.ShowDialog();
            if (result == DialogResult.OK)
            {
                dirText.Text = folderDlg.SelectedPath;
                Environment.SpecialFolder root = folderDlg.RootFolder;
            }
        }

        private void OpenFileButton_Click(object sender, EventArgs e)
        {
            OpenFileDialog openFile = new OpenFileDialog();
            openFile.Filter = "XML File (*.xml) | *.xml| (*.*)|*.*";

            if (openFile.ShowDialog() == DialogResult.OK)
            {
                
                
                NPCData npcData = new NPCData();
                LineHandling.SetUpDataBaseFromFile(FileHandler.LoadXMLFile(openFile.FileName, out npcData));
                idText.Text = npcData.npcID.ToString();
                nameText.Text = npcData.npcName;
                dirText.Text = Path.GetDirectoryName(openFile.FileName);
                fileNameText.Text = Path.GetFileNameWithoutExtension(openFile.FileName);

            }
        }
    }
}
